#!/usr/bin/env python3
import json, re, sys, math, hashlib
from pathlib import Path
import pandas as pd
import numpy as np

ROOT = Path(__file__).resolve().parent

def sha256_file(p: Path) -> str:
    h = hashlib.sha256()
    with open(p, "rb") as f:
        for chunk in iter(lambda: f.read(8192), b""):
            h.update(chunk)
    return h.hexdigest()

def runs_test_p(residuals: np.ndarray) -> float:
    med = np.median(residuals)
    s = np.array([1 if r >= med else 0 for r in residuals], dtype=int)
    n1 = int(s.sum())
    n2 = int(len(s) - n1)
    if n1 == 0 or n2 == 0:
        return 1.0
    runs = 1 + int(np.sum(s[1:] != s[:-1]))
    mu = 1 + 2*n1*n2/(n1+n2)
    var = (2*n1*n2*(2*n1*n2 - n1 - n2))/(((n1+n2)**2)*(n1+n2-1))
    if var <= 0: return 1.0
    z = (runs - mu)/math.sqrt(var)
    # two-sided normal tail via erf
    from math import erf, sqrt
    p = 2 * (1 - 0.5*(1+erf(abs(z)/sqrt(2))))
    return float(p)

def linear_fit(x, y):
    x = np.asarray(x); y = np.asarray(y)
    X = np.vstack([x, np.ones_like(x)]).T
    coef, _, _, _ = np.linalg.lstsq(X, y, rcond=None)
    A, B = coef[0], coef[1]
    yhat = A*x + B
    resid = y - yhat
    ss_res = float(np.sum(resid**2))
    ss_tot = float(np.sum((y - np.mean(y))**2))
    R2 = 1 - ss_res/ss_tot if ss_tot > 0 else 1.0
    return float(A), float(B), yhat, resid, float(R2)

def main():
    # Load tolerances and windows from study.yaml (simple parse)
    study = (ROOT/"manifests"/"study.yaml").read_text()
    def get_float(key, default=None):
        m = re.search(rf"{key}:\s*([0-9.eE+-]+)", study)
        return float(m.group(1)) if m else default
    R2_min      = get_float("R2_min", 0.95)
    tau_mesh    = get_float("tau_mesh", 0.02)
    tau_ratio   = get_float("tau_ratio", 0.02)
    tau_rc      = get_float("tau_rc", 0.03)
    corr_min    = 0.99

    # Hash consistency
    manifest_txt = (ROOT/"manifests"/"manifest.yaml").read_text()
    m = re.search(r'dag_hash:\s*"(.*?)"', manifest_txt)
    dag_hash_manifest = m.group(1) if m else ""
    manifest_hash = sha256_file(ROOT/"manifests"/"manifest.yaml")
    catalog_hash  = (ROOT/"catalog"/"catalog_hash.txt").read_text().strip()
    hinge_hash    = (ROOT/"hinges"/"hinge_hash.txt").read_text().strip()

    # Prereg & catalog
    prereg = json.loads((ROOT/"prereg"/"preregister.json").read_text())
    ratio_ci = prereg["ratios"]["K_delta_t"]
    K_lo, K_hi = float(ratio_ci["ci_lo"]), float(ratio_ci["ci_hi"])
    catalog = json.loads((ROOT/"catalog"/"alphaP_catalog.json").read_text())
    Sring = float(catalog["Sring_mean"])
    R_oplus = 1.0
    # Hinges
    hinges = json.loads((ROOT/"hinges"/"hinges.json").read_text())
    c_pred = float(hinges["c_pred"])

    # Load Shapiro CSV
    df = pd.read_csv(ROOT/"runs"/"shapiro_delay.csv")
    # Check per-row hashes match shipped files
    assert (df["manifest_hash"] == manifest_hash).all(), "manifest_hash mismatch in CSV rows"
    assert (df["catalog_hash"]  == catalog_hash).all(),  "catalog_hash mismatch in CSV rows"
    assert (df["hinge_hash"]    == hinge_hash).all(),    "hinge_hash mismatch in CSV rows"
    assert (df["dag_hash"]      == dag_hash_manifest).all(), "dag_hash mismatch in CSV rows"

    # Recompute slopes from dt_diff vs logLb per mesh and verify K_delta_t
    for mesh in ["h","h2"]:
        sub = df[df["mesh"]==mesh]
        A,B,yhat,resid,R2 = linear_fit(sub["logLb"].values, sub["dt_diff"].values)
        assert R2 >= R2_min, f"{mesh}: R^2 {R2:.6f} < {R2_min}"
        p_runs = runs_test_p(resid)
        assert p_runs >= 0.05, f"{mesh}: residual runs-test p {p_runs:.3f} < 0.05"
        K = (c_pred*A)/(Sring*R_oplus)
        assert K_lo <= K <= K_hi, f"{mesh}: KΔt {K:.3f} outside [{K_lo:.3f},{K_hi:.3f}]"

    # Mesh certification
    mc = json.loads((ROOT/"audits"/"mesh_cert.json").read_text())
    assert mc["passed"] is True, "mesh_cert audit failed"
    assert float(mc["amp_rel_delta"]) <= tau_mesh + 1e-12, "amp_rel_delta > tau_mesh"
    assert float(mc["shape_corr"]) >= corr_min - 1e-12, "shape_corr < 0.99"

    # Re-centering audit
    rc = json.loads((ROOT/"audits"/"recentering_audit.json").read_text())
    assert rc["passed"] is True, "recentering audit failed"

    # Must-pass audits
    audits = {
        "neutrality_audit.json": lambda a: a["passed"] is True,
        "sr_fit.json":           lambda a: a["passed"] is True and a["cone_leakage"]==0,
        "isotropy.json":         lambda a: a["passed"] is True,
        "monotone.json":         lambda a: a["passed"] is True,
        "nosig.json":            lambda a: a["passed"] is True,
        "diag_leak.json":        lambda a: a["passed"] is True and a["taint_free"] is True,
        "rng_ties.json":         lambda a: a["passed"] is True and a["tie_count"] >= 16000,
    }
    for name, pred in audits.items():
        obj = json.loads((ROOT/"audits"/name).read_text())
        assert pred(obj), f"audit {name} failed"

    # Row-level claimability
    assert df["claimable"].all(), "Some Shapiro rows are not claimable"
    assert (df["reason"].fillna("")=="").all(), "Some Shapiro rows have failure reasons"

    print("✅ Verification PASSED: all checks green.")

if __name__ == "__main__":
    try:
        main()
    except AssertionError as e:
        print("❌ Verification FAILED:", e)
        sys.exit(2)
    except Exception as e:
        print("❌ Error:", e)
        sys.exit(3)
